# Load Packages

library(tidyverse)
library(data.table)
library(ggplot2)
library(tictoc)
library(fixest)
library(arrow)
library(qs)

data <- fread("P:/Company Tax/Data/EUM/bas_eum_reg_data.csv")

# Filtering for time
DnD15 <- data[tsid >= 15 & tsid <= 16]
DnD16 <- data[tsid >= 16 & tsid <= 17]
DnD17 <- data[tsid >= 17 & tsid <= 18]
DnD18 <- data[tsid >= 18 & tsid <= 19]

# Creating Treated Variables
DnD15 <- DnD15[, treated := ifelse(total_comp_income < 2 * 10^6, 1, 0)] # Lower tax rate if turnover < $2m
DnD15 <- DnD15[, treated := ifelse(base_rate_entity == 0 & tsid >= 18, 0, treated)] # If they are not a base rate entity, they don't get the lower tax rate
DnD15 <- DnD15[, policy_period := ifelse(tsid >= 16, 1, 0)] # tsid is the last 2 digits of a financial year

DnD16 <- DnD16[, treated := ifelse(total_comp_income < 10 * 10^6 & total_comp_income >= 2 * 10^6 , 1, 0)] # Excluding those that were previously treated
DnD16 <- DnD16[, treated := ifelse(base_rate_entity == 0 & tsid >= 18, 0, treated)] # If they are not a base rate entity, they don't get the lower tax rate
DnD16 <- DnD16[, policy_period := ifelse(tsid >= 17, 1, 0)]
DnD16 <- DnD16[total_comp_income >= 2 * 10^6] # Exclude companies that are already treated

DnD17 <- DnD17[, treated := ifelse(total_comp_income < 25 * 10^6 & total_comp_income >= 10 * 10^6 , 1, 0)] # Excluding those that were previously treated
DnD17 <- DnD17[, treated := ifelse(base_rate_entity == 0 & tsid >= 18, 0, treated)] # If they are not a base rate entity, they don't get the lower tax rate
DnD17 <- DnD17[, policy_period := ifelse(tsid >= 18, 1, 0)] # tsid is the last 2 digits of a financial year
DnD17 <- DnD17[turnover >= 10 * 10^6] # After 2017, BAS data is unreliable for businesses with turnover below $10M since they don't have to report everything
DnD17 <- DnD17[total_comp_income >= 10 * 10^6] # Exclude companies that are already treated

DnD18 <- DnD18[, treated := ifelse(total_comp_income < 50 * 10^6 & total_comp_income >= 25 * 10^6 , 1, 0)] # Excluding those that were previously treated
DnD18 <- DnD18[, treated := ifelse(base_rate_entity == 0 & tsid >= 18, 0, treated)] # If they are not a base rate entity, they don't get the lower tax rate
DnD18 <- DnD18[, policy_period := ifelse(tsid >= 19, 1, 0)] # tsid is the last 2 digits of a financial year
DnD18 <- DnD18[turnover >= 10 * 10^6] # After 2017, BAS data is unreliable for businesses with turnover below $10M since they don't have to report everything
DnD18 <- DnD18[total_comp_income >= 25 * 10^6] # Exclude companies that are already treated

# Only include companies for DnD
DnD15 <- DnD15[company_flag == 1]
DnD16 <- DnD16[company_flag == 1]
DnD17 <- DnD17[company_flag == 1]
DnD18 <- DnD18[company_flag == 1]

# LIQUIDITY DDD
# Estimate the effect of the 2015, 2016, 2017 and 2018 Tax Cut on the extensive margin
# Creating Log Odds Ratio
DnD15 <- DnD15[, `:=`(
  num_comp_by_industry = sum(!is.na(capex)),
  invs_size_ind = sum(invest_dummy)
), by = .(treated, industry_simple, lqdty_flag, tsid)]

DnD15 <- DnD15[, invs_size_ind_share := invs_size_ind / num_comp_by_industry] # Compute share of firms investing
DnD15 <- DnD15[, log_odds := fcase(
  invs_size_ind_share == 1 | log(invs_size_ind_share / (1 - invs_size_ind_share)) > 4, 4,
  invs_size_ind_share == 0 | log(invs_size_ind_share / (1 - invs_size_ind_share)) < -4, -4,
  invs_size_ind_share > 0 & invs_size_ind_share < 1, log(invs_size_ind_share / (1 - invs_size_ind_share))
  )] 

model_DnD15_Ext <- feols(
  log_odds ~ treated * policy_period * lqdty_flag + age + fte_log + log_tot_inc + foreign_flag | 
    # FIXED EFFECTS
    industry_simple^tsid_factor + 
    bg_id
  ,
  cluster = ~ industry_simple,
  data = DnD15
)

# Creating Log Odds Ratio
DnD16[, `:=`(
  num_comp_by_industry = sum(!is.na(capex)),
  invs_size_ind = sum(invest_dummy)
), by = .(treated, industry_simple, lqdty_flag, tsid)]

DnD16 <- DnD16[, invs_size_ind_share := invs_size_ind / num_comp_by_industry] # Compute share of firms investing
DnD16 <- DnD16[, log_odds := fcase(
  invs_size_ind_share == 1 | log(invs_size_ind_share / (1 - invs_size_ind_share)) > 4, 4,
  invs_size_ind_share == 0 | log(invs_size_ind_share / (1 - invs_size_ind_share)) < -4, -4,
  invs_size_ind_share > 0 & invs_size_ind_share < 1, log(invs_size_ind_share / (1 - invs_size_ind_share))
)] 

model_DnD16_Ext <- feols(
  log_odds ~ treated * policy_period * lqdty_flag + age + fte_log + log_tot_inc + foreign_flag | 
    # FIXED EFFECTS
    industry_simple^tsid_factor + 
    bg_id
  ,
  cluster = ~ industry_simple,
  data = DnD16
)

# Creating Log Odds Ratio
DnD17[, `:=`(
  num_comp_by_industry = sum(!is.na(capex)),
  invs_size_ind = sum(invest_dummy)
), by = .(treated, industry_simple, lqdty_flag, tsid)]

DnD17 <- DnD17[, invs_size_ind_share := invs_size_ind / num_comp_by_industry] # Compute share of firms investing
DnD17 <- DnD17[, log_odds := fcase(
  invs_size_ind_share == 1 | log(invs_size_ind_share / (1 - invs_size_ind_share)) > 4, 4,
  invs_size_ind_share == 0 | log(invs_size_ind_share / (1 - invs_size_ind_share)) < -4, -4,
  invs_size_ind_share > 0 & invs_size_ind_share < 1, log(invs_size_ind_share / (1 - invs_size_ind_share))
)]  

model_DnD17_Ext <- feols(
  log_odds ~ treated * policy_period * lqdty_flag + age + fte_log + log_tot_inc + foreign_flag | 
    # FIXED EFFECTS
    industry_simple^tsid_factor + 
    bg_id
  ,
  cluster = ~ industry_simple,
  data = DnD17
)

# Creating Log Odds Ratio
DnD18[, `:=`(
  num_comp_by_industry = sum(!is.na(capex)),
  invs_size_ind = sum(invest_dummy)
  ), by = .(treated, industry_simple, lqdty_flag, tsid)]

DnD18 <- DnD18[, invs_size_ind_share := invs_size_ind / num_comp_by_industry] # Compute share of firms investing
DnD18 <- DnD18[, log_odds := fcase(
  invs_size_ind_share == 1 | log(invs_size_ind_share / (1 - invs_size_ind_share)) > 4, 4,
  invs_size_ind_share == 0 | log(invs_size_ind_share / (1 - invs_size_ind_share)) < -4, -4,
  invs_size_ind_share > 0 & invs_size_ind_share < 1, log(invs_size_ind_share / (1 - invs_size_ind_share))
)] 

model_DnD18_Ext <- feols(
  log_odds ~ treated * policy_period * lqdty_flag + age + fte_log + log_tot_inc + foreign_flag | 
    # FIXED EFFECTS
    industry_simple^tsid_factor + 
    bg_id
  ,
  cluster = ~ industry_simple,
  data = DnD18
)

# Saving only coefficient outputs and not fixed effects
tidy_model_DnD15_Ext <- broom::tidy(model_DnD15_Ext)
tidy_model_DnD16_Ext <- broom::tidy(model_DnD16_Ext)
tidy_model_DnD17_Ext <- broom::tidy(model_DnD17_Ext)
tidy_model_DnD18_Ext <- broom::tidy(model_DnD18_Ext)

qs::qsave(tidy_model_DnD15_Ext, "P:/Company Tax/Econometrics for Clearance/Models to Clear/DDD With Controls/BAS_LQDTY_DDD_15_Ext.qs")
qs::qsave(tidy_model_DnD16_Ext, "P:/Company Tax/Econometrics for Clearance/Models to Clear/DDD With Controls/BAS_LQDTY_DDD_16_Ext.qs")
qs::qsave(tidy_model_DnD17_Ext, "P:/Company Tax/Econometrics for Clearance/Models to Clear/DDD With Controls/BAS_LQDTY_DDD_17_Ext.qs")
qs::qsave(tidy_model_DnD18_Ext, "P:/Company Tax/Econometrics for Clearance/Models to Clear/DDD With Controls/BAS_LQDTY_DDD_18_Ext.qs")

model_names <- c("model_DnD15_Ext", "model_DnD16_Ext", "model_DnD17_Ext", "model_DnD18_Ext")

BAS_LQDTY_Extensive_num_obs_list <- data.frame(
  Model = model_names,
  Number_of_Observations = sapply(mget(model_names), nobs)
  ) %>% 
  mutate(Model = paste0("BAS_LQDTY_", Model))

# Foreign Flag Control
# Estimate the effect of the 2015, 2016, 2017 and 2018 Tax Cut on the extensive margin

# Creating Log Odds Ratio
DnD15 <- DnD15[, `:=`(
  num_comp_by_industry = sum(!is.na(capex)),
  invs_size_ind = sum(invest_dummy)
), by = .(treated, industry_simple, foreign_flag, tsid)]

DnD15 <- DnD15[, invs_size_ind_share := invs_size_ind / num_comp_by_industry] # Compute share of firms investing
DnD15 <- DnD15[, log_odds := fcase(
  invs_size_ind_share == 1 | log(invs_size_ind_share / (1 - invs_size_ind_share)) > 4, 4,
  invs_size_ind_share == 0 | log(invs_size_ind_share / (1 - invs_size_ind_share)) < -4, -4,
  invs_size_ind_share > 0 & invs_size_ind_share < 1, log(invs_size_ind_share / (1 - invs_size_ind_share))
)] 

model_DnD15_Ext <- feols(
  log_odds ~ treated * policy_period * foreign_flag + log_lqdty + age + fte_log + log_tot_inc | 
    # FIXED EFFECTS
    industry_simple^tsid_factor + 
    bg_id
  ,
  cluster = ~ industry_simple,
  data = DnD15
)

# Creating Log Odds Ratio
DnD16[, `:=`(
  num_comp_by_industry = sum(!is.na(capex)),
  invs_size_ind = sum(invest_dummy)
), by = .(treated, industry_simple, foreign_flag, tsid)]

DnD16 <- DnD16[, invs_size_ind_share := invs_size_ind / num_comp_by_industry] # Compute share of firms investing
DnD16 <- DnD16[, log_odds := fcase(
  invs_size_ind_share == 1 | log(invs_size_ind_share / (1 - invs_size_ind_share)) > 4, 4,
  invs_size_ind_share == 0 | log(invs_size_ind_share / (1 - invs_size_ind_share)) < -4, -4,
  invs_size_ind_share > 0 & invs_size_ind_share < 1, log(invs_size_ind_share / (1 - invs_size_ind_share))
  )] 

model_DnD16_Ext <- feols(
  log_odds ~ treated * policy_period * foreign_flag + age + fte_log + log_tot_inc + log_lqdty | 
    # FIXED EFFECTS
    industry_simple^tsid_factor + 
    bg_id
  ,
  cluster = ~ industry_simple,
  data = DnD16
)

# Creating Log Odds Ratio
DnD17[, `:=`(
  num_comp_by_industry = sum(!is.na(capex)),
  invs_size_ind = sum(invest_dummy)
), by = .(treated, industry_simple, foreign_flag, tsid)]

DnD17 <- DnD17[, invs_size_ind_share := invs_size_ind / num_comp_by_industry] # Compute share of firms investing
DnD17 <- DnD17[, log_odds := fcase(
  invs_size_ind_share == 1 | log(invs_size_ind_share / (1 - invs_size_ind_share)) > 4, 4,
  invs_size_ind_share == 0 | log(invs_size_ind_share / (1 - invs_size_ind_share)) < -4, -4,
  invs_size_ind_share > 0 & invs_size_ind_share < 1, log(invs_size_ind_share / (1 - invs_size_ind_share))
)] 

model_DnD17_Ext <- feols(
  log_odds ~ treated * policy_period * foreign_flag + age + fte_log + log_tot_inc + log_lqdty | 
    # FIXED EFFECTS
    industry_simple^tsid_factor + 
    bg_id
  ,
  cluster = ~ industry_simple,
  data = DnD17
)

# Creating Log Odds Ratio
DnD18[, `:=`(
  num_comp_by_industry = sum(!is.na(capex)),
  invs_size_ind = sum(invest_dummy)
), by = .(treated, industry_simple, foreign_flag, tsid)]

DnD18 <- DnD18[, invs_size_ind_share := invs_size_ind / num_comp_by_industry] # Compute share of firms investing
DnD18 <- DnD18[, log_odds := fcase(
  invs_size_ind_share == 1 | log(invs_size_ind_share / (1 - invs_size_ind_share)) > 4, 4,
  invs_size_ind_share == 0 | log(invs_size_ind_share / (1 - invs_size_ind_share)) < -4, -4,
  invs_size_ind_share > 0 & invs_size_ind_share < 1, log(invs_size_ind_share / (1 - invs_size_ind_share))
)] 

model_DnD18_Ext <- feols(
  log_odds ~ treated * policy_period * foreign_flag + age + fte_log + log_tot_inc + log_lqdty | 
    # FIXED EFFECTS
    industry_simple^tsid_factor + 
    bg_id
  ,
  cluster = ~ industry_simple,
  data = DnD18
)

# Saving only coefficient outputs and not fixed effects
tidy_model_DnD15_Ext <- broom::tidy(model_DnD15_Ext)
tidy_model_DnD16_Ext <- broom::tidy(model_DnD16_Ext)
tidy_model_DnD17_Ext <- broom::tidy(model_DnD17_Ext)
tidy_model_DnD18_Ext <- broom::tidy(model_DnD18_Ext)

qs::qsave(tidy_model_DnD15_Ext, "P:/Company Tax/Econometrics for Clearance/Models to Clear/DDD With Controls/BAS_FRGN_DDD_15_Ext.qs")
qs::qsave(tidy_model_DnD16_Ext, "P:/Company Tax/Econometrics for Clearance/Models to Clear/DDD With Controls/BAS_FRGN_DDD_16_Ext.qs")
qs::qsave(tidy_model_DnD17_Ext, "P:/Company Tax/Econometrics for Clearance/Models to Clear/DDD With Controls/BAS_FRGN_DDD_17_Ext.qs")
qs::qsave(tidy_model_DnD18_Ext, "P:/Company Tax/Econometrics for Clearance/Models to Clear/DDD With Controls/BAS_FRGN_DDD_18_Ext.qs")

model_names <- c("model_DnD15_Ext", "model_DnD16_Ext", "model_DnD17_Ext", "model_DnD18_Ext")

BAS_FRGN_Extensive_num_obs_list <- data.frame(
  Model = model_names,
  Number_of_Observations = sapply(mget(model_names), nobs)
) %>% 
  mutate(Model = paste0("BAS_FRGN_", Model))

# Intensive Regressions
# Remove CAPEX = 0
DnD15 <- DnD15[capex != 0]
DnD16 <- DnD16[capex != 0]
DnD17 <- DnD17[capex != 0]
DnD18 <- DnD18[capex != 0]

## LIQUIDITY DDD
model_DnD15_Int <- feols(
  capex_log ~ treated * policy_period * lqdty_flag  + age + fte_log + log_tot_inc + foreign_flag | 
    # FIXED EFFECTS
    industry_simple^tsid_factor + 
    bg_id,
  cluster = ~ bg_id, 
  data = DnD15)

model_DnD16_Int <- feols(
  capex_log ~ treated * policy_period * lqdty_flag + age + fte_log + log_tot_inc + foreign_flag| 
    # FIXED EFFECTS
    industry_simple^tsid_factor + bg_id,
  cluster = ~ bg_id, 
  data = DnD16)

model_DnD17_Int <- feols(
  capex_log ~ treated * policy_period * lqdty_flag + age + fte_log + log_tot_inc + foreign_flag | 
    # FIXED EFFECTS
    industry_simple^tsid_factor + bg_id,
  cluster = ~ bg_id, 
  data = DnD17)

model_DnD18_Int <- feols(
  capex_log ~ treated * policy_period * lqdty_flag + age + fte_log + log_tot_inc + foreign_flag | 
    # FIXED EFFECTS
    industry_simple^tsid_factor + bg_id,
  cluster = ~ bg_id, 
  data = DnD18)

# Saving only coefficient outputs and not fixed effects
tidy_model_DnD15_Int <- broom::tidy(model_DnD15_Int)
tidy_model_DnD16_Int <- broom::tidy(model_DnD16_Int)
tidy_model_DnD17_Int <- broom::tidy(model_DnD17_Int)
tidy_model_DnD18_Int <- broom::tidy(model_DnD18_Int)

qs::qsave(tidy_model_DnD15_Int, "P:/Company Tax/Econometrics for Clearance/Models to Clear/DDD With Controls/BAS_LQDTY_DDD_15_Int.qs")
qs::qsave(tidy_model_DnD16_Int, "P:/Company Tax/Econometrics for Clearance/Models to Clear/DDD With Controls/BAS_LQDTY_DDD_16_Int.qs")
qs::qsave(tidy_model_DnD17_Int, "P:/Company Tax/Econometrics for Clearance/Models to Clear/DDD With Controls/BAS_LQDTY_DDD_17_Int.qs")
qs::qsave(tidy_model_DnD18_Int, "P:/Company Tax/Econometrics for Clearance/Models to Clear/DDD With Controls/BAS_LQDTY_DDD_18_Int.qs")

model_names <- c(
  "model_DnD15_Int", "model_DnD16_Int", "model_DnD17_Int", "model_DnD18_Int")

BAS_LQDTY_Intensive_num_obs_list <- data.frame(
  Model = model_names,
  Number_of_Observations = sapply(mget(model_names), nobs)
) %>% 
  mutate(Model = paste0("BAS_LQDTY_", Model))

## FOREIGN FLAG DDD

model_DnD15_Int <- feols(
  capex_log ~ treated * policy_period * foreign_flag  + age + fte_log + log_tot_inc + log_lqdty | 
    # FIXED EFFECTS
    industry_simple^tsid_factor + 
    bg_id,
  cluster = ~ bg_id, 
  data = DnD15)

model_DnD16_Int <- feols(
  capex_log ~ treated * policy_period * foreign_flag + age + fte_log + log_tot_inc + log_lqdty | 
    # FIXED EFFECTS
    industry_simple^tsid_factor + bg_id,
  cluster = ~ bg_id, 
  data = DnD16)

model_DnD17_Int <- feols(
  capex_log ~ treated * policy_period * foreign_flag  + age + fte_log + log_tot_inc + log_lqdty | 
    # FIXED EFFECTS
    industry_simple^tsid_factor + bg_id,
  cluster = ~ bg_id, 
  data = DnD17)

model_DnD18_Int <- feols(
  capex_log ~ treated * policy_period * foreign_flag + age + fte_log + log_tot_inc + log_lqdty | 
    # FIXED EFFECTS
    industry_simple^tsid_factor + bg_id,
  cluster = ~ bg_id, 
  data = DnD18)

# Saving only coefficient outputs and not fixed effects
tidy_model_DnD15_Int <- broom::tidy(model_DnD15_Int)
tidy_model_DnD16_Int <- broom::tidy(model_DnD16_Int)
tidy_model_DnD17_Int <- broom::tidy(model_DnD17_Int)
tidy_model_DnD18_Int <- broom::tidy(model_DnD18_Int)

qs::qsave(tidy_model_DnD15_Int, "P:/Company Tax/Econometrics for Clearance/Models to Clear/DDD With Controls/BAS_FRGN_DDD_15_Int.qs")
qs::qsave(tidy_model_DnD16_Int, "P:/Company Tax/Econometrics for Clearance/Models to Clear/DDD With Controls/BAS_FRGN_DDD_16_Int.qs")
qs::qsave(tidy_model_DnD17_Int, "P:/Company Tax/Econometrics for Clearance/Models to Clear/DDD With Controls/BAS_FRGN_DDD_17_Int.qs")
qs::qsave(tidy_model_DnD18_Int, "P:/Company Tax/Econometrics for Clearance/Models to Clear/DDD With Controls/BAS_FRGN_DDD_18_Int.qs")

model_names <- c(
  "model_DnD15_Int", "model_DnD16_Int", "model_DnD17_Int", "model_DnD18_Int")

BAS_FRGN_Intensive_num_obs_list <- data.frame(
  Model = model_names,
  Number_of_Observations = sapply(mget(model_names), nobs)
) %>% 
  mutate(Model = paste0("BAS_FRGN_", Model))

# CAPEX ------------------------------------------------------------------------

data <- fread("P:/Company Tax/Data/EUM/capex_eum_reg_data.csv")

# Filtering for time
DnD15 <- data[tsid >= 15 & tsid <= 16]
DnD16 <- data[tsid >= 16 & tsid <= 17]
DnD17 <- data[tsid >= 17 & tsid <= 18]
DnD18 <- data[tsid >= 18 & tsid <= 19]

# Creating Treated Variables
DnD15 <- DnD15[, treated := ifelse(total_comp_income < 2 * 10^6, 1, 0)] # Lower tax rate if turnover < $2m
DnD15 <- DnD15[, treated := ifelse(base_rate_entity == 0 & tsid >= 18, 0, treated)] # If they are not a CAPEXe rate entity, they don't get the lower tax rate
DnD15 <- DnD15[, policy_period := ifelse(tsid >= 16, 1, 0)] # tsid is the last 2 digits of a financial year

DnD16 <- DnD16[, treated := ifelse(total_comp_income < 10 * 10^6 & total_comp_income >= 2 * 10^6 , 1, 0)] # Excluding those that were previously treated
DnD16 <- DnD16[, treated := ifelse(base_rate_entity == 0 & tsid >= 18, 0, treated)] # If they are not a CAPEXe rate entity, they don't get the lower tax rate
DnD16 <- DnD16[, policy_period := ifelse(tsid >= 17, 1, 0)]
DnD16 <- DnD16[total_comp_income >= 2 * 10^6] # Exclude companies that are already treated

DnD17 <- DnD17[, treated := ifelse(total_comp_income < 25 * 10^6 & total_comp_income >= 10 * 10^6 , 1, 0)] # Excluding those that were previously treated
DnD17 <- DnD17[, treated := ifelse(base_rate_entity == 0 & tsid >= 18, 0, treated)] # If they are not a CAPEXe rate entity, they don't get the lower tax rate
DnD17 <- DnD17[, policy_period := ifelse(tsid >= 18, 1, 0)] # tsid is the last 2 digits of a financial year
DnD17 <- DnD17[total_comp_income >= 10 * 10^6] # Exclude companies that are already treated

DnD18 <- DnD18[, treated := ifelse(total_comp_income < 50 * 10^6 & total_comp_income >= 25 * 10^6 , 1, 0)] # Excluding those that were previously treated
DnD18 <- DnD18[, treated := ifelse(base_rate_entity == 0 & tsid >= 18, 0, treated)] # If they are not a CAPEXe rate entity, they don't get the lower tax rate
DnD18 <- DnD18[, policy_period := ifelse(tsid >= 19, 1, 0)] # tsid is the last 2 digits of a financial year
DnD18 <- DnD18[total_comp_income >= 25 * 10^6] # Exclude companies that are already treated


# Extensive Margin
# LIQUIDITY DDD

# Creating Log Odds Ratio
DnD15 <- DnD15[, `:=`(
  num_comp_by_industry = sum(!is.na(capex)),
  invs_size_ind = sum(invest_dummy)
), by = .(treated, industry_simple, lqdty_flag, tsid)]

DnD15 <- DnD15[, invs_size_ind_share := invs_size_ind / num_comp_by_industry] # Compute share of firms investing
DnD15 <- DnD15[, log_odds := fcase(
  invs_size_ind_share == 1 | log(invs_size_ind_share / (1 - invs_size_ind_share)) > 4, 4,
  invs_size_ind_share == 0 | log(invs_size_ind_share / (1 - invs_size_ind_share)) < -4, -4,
  invs_size_ind_share > 0 & invs_size_ind_share < 1, log(invs_size_ind_share / (1 - invs_size_ind_share))
)] 

model_DnD15_Ext <- feols(
  log_odds ~ treated * policy_period * lqdty_flag + age + fte_log + log_tot_inc + foreign_flag | 
    # FIXED EFFECTS
    industry_simple^tsid_factor + 
    bg_id
  ,
  cluster = ~ industry_simple,
  data = DnD15
)

# Creating Log Odds Ratio
DnD16[, `:=`(
  num_comp_by_industry = sum(!is.na(capex)),
  invs_size_ind = sum(invest_dummy)
), by = .(treated, industry_simple, lqdty_flag, tsid)]

DnD16 <- DnD16[, invs_size_ind_share := invs_size_ind / num_comp_by_industry] # Compute share of firms investing
DnD16 <- DnD16[, log_odds := fcase(
  invs_size_ind_share == 1 | log(invs_size_ind_share / (1 - invs_size_ind_share)) > 4, 4,
  invs_size_ind_share == 0 | log(invs_size_ind_share / (1 - invs_size_ind_share)) < -4, -4,
  invs_size_ind_share > 0 & invs_size_ind_share < 1, log(invs_size_ind_share / (1 - invs_size_ind_share))
)] 

model_DnD16_Ext <- feols(
  log_odds ~ treated * policy_period * lqdty_flag + age + fte_log + log_tot_inc + foreign_flag | 
    # FIXED EFFECTS
    industry_simple^tsid_factor + 
    bg_id
  ,
  cluster = ~ industry_simple,
  data = DnD16
)

# Creating Log Odds Ratio
DnD17[, `:=`(
  num_comp_by_industry = sum(!is.na(capex)),
  invs_size_ind = sum(invest_dummy)
), by = .(treated, industry_simple, lqdty_flag, tsid)]

DnD17 <- DnD17[, invs_size_ind_share := invs_size_ind / num_comp_by_industry] # Compute share of firms investing
DnD17 <- DnD17[, log_odds := fcase(
  invs_size_ind_share == 1 | log(invs_size_ind_share / (1 - invs_size_ind_share)) > 4, 4,
  invs_size_ind_share == 0 | log(invs_size_ind_share / (1 - invs_size_ind_share)) < -4, -4,
  invs_size_ind_share > 0 & invs_size_ind_share < 1, log(invs_size_ind_share / (1 - invs_size_ind_share))
)] 

model_DnD17_Ext <- feols(
  log_odds ~ treated * policy_period * lqdty_flag + age + fte_log + log_tot_inc + foreign_flag | 
    # FIXED EFFECTS
    industry_simple^tsid_factor + 
    bg_id
  ,
  cluster = ~ industry_simple,
  data = DnD17
)

# Creating Log Odds Ratio
DnD18[, `:=`(
  num_comp_by_industry = sum(!is.na(capex)),
  invs_size_ind = sum(invest_dummy)
), by = .(treated, industry_simple, lqdty_flag, tsid)]

DnD18 <- DnD18[, invs_size_ind_share := invs_size_ind / num_comp_by_industry] # Compute share of firms investing
DnD18 <- DnD18[, log_odds := fcase(
  invs_size_ind_share == 1 | log(invs_size_ind_share / (1 - invs_size_ind_share)) > 4, 4,
  invs_size_ind_share == 0 | log(invs_size_ind_share / (1 - invs_size_ind_share)) < -4, -4,
  invs_size_ind_share > 0 & invs_size_ind_share < 1, log(invs_size_ind_share / (1 - invs_size_ind_share))
)] 

model_DnD18_Ext <- feols(
  log_odds ~ treated * policy_period * lqdty_flag + age + fte_log + log_tot_inc + foreign_flag | 
    # FIXED EFFECTS
    industry_simple^tsid_factor + 
    bg_id
  ,
  cluster = ~ industry_simple,
  data = DnD18
)

# Saving only coefficient outputs and not fixed effects
tidy_model_DnD15_Ext <- broom::tidy(model_DnD15_Ext)
tidy_model_DnD16_Ext <- broom::tidy(model_DnD16_Ext)
tidy_model_DnD17_Ext <- broom::tidy(model_DnD17_Ext)
tidy_model_DnD18_Ext <- broom::tidy(model_DnD18_Ext)

qs::qsave(tidy_model_DnD15_Ext, "P:/Company Tax/Econometrics for Clearance/Models to Clear/DDD With Controls/CAPEX_LQDTY_DDD_15_Ext.qs")
qs::qsave(tidy_model_DnD16_Ext, "P:/Company Tax/Econometrics for Clearance/Models to Clear/DDD With Controls/CAPEX_LQDTY_DDD_16_Ext.qs")
qs::qsave(tidy_model_DnD17_Ext, "P:/Company Tax/Econometrics for Clearance/Models to Clear/DDD With Controls/CAPEX_LQDTY_DDD_17_Ext.qs")
qs::qsave(tidy_model_DnD18_Ext, "P:/Company Tax/Econometrics for Clearance/Models to Clear/DDD With Controls/CAPEX_LQDTY_DDD_18_Ext.qs")

model_names <- c(
  "model_DnD15_Ext", "model_DnD16_Ext", "model_DnD17_Ext", "model_DnD18_Ext")

CAPEX_LQDTY_DD_Extensive_num_obs_list <- data.frame(
  Model = model_names,
  Number_of_Observations = sapply(mget(model_names), nobs)
) %>% 
  mutate(Model = paste0("CAPEX_LQDTY_", Model))


# Foreign Flag Control

# Creating Log Odds Ratio
DnD15 <- DnD15[, `:=`(
  num_comp_by_industry = sum(!is.na(capex)),
  invs_size_ind = sum(invest_dummy)
), by = .(treated, industry_simple, foreign_flag, tsid)]

DnD15 <- DnD15[, invs_size_ind_share := invs_size_ind / num_comp_by_industry] # Compute share of firms investing
DnD15 <- DnD15[, log_odds := fcase(
  invs_size_ind_share == 1 | log(invs_size_ind_share / (1 - invs_size_ind_share)) > 4, 4,
  invs_size_ind_share == 0 | log(invs_size_ind_share / (1 - invs_size_ind_share)) < -4, -4,
  invs_size_ind_share > 0 & invs_size_ind_share < 1, log(invs_size_ind_share / (1 - invs_size_ind_share))
)] 

model_DnD15_Ext <- feols(
  log_odds ~ treated * policy_period * foreign_flag + log_lqdty + age + fte_log + log_tot_inc | 
    # FIXED EFFECTS
    industry_simple^tsid_factor + 
    bg_id
  ,
  cluster = ~ industry_simple,
  data = DnD15
)

# Creating Log Odds Ratio
DnD16[, `:=`(
  num_comp_by_industry = sum(!is.na(capex)),
  invs_size_ind = sum(invest_dummy)
), by = .(treated, industry_simple, foreign_flag, tsid)]

DnD16 <- DnD16[, invs_size_ind_share := invs_size_ind / num_comp_by_industry] # Compute share of firms investing
DnD16 <- DnD16[, log_odds := fcase(
  invs_size_ind_share == 1 | log(invs_size_ind_share / (1 - invs_size_ind_share)) > 4, 4,
  invs_size_ind_share == 0 | log(invs_size_ind_share / (1 - invs_size_ind_share)) < -4, -4,
  invs_size_ind_share > 0 & invs_size_ind_share < 1, log(invs_size_ind_share / (1 - invs_size_ind_share))
)] 

model_DnD16_Ext <- feols(
  log_odds ~ treated * policy_period * foreign_flag + age + fte_log + log_tot_inc + log_lqdty | 
    # FIXED EFFECTS
    industry_simple^tsid_factor + 
    bg_id
  ,
  cluster = ~ industry_simple,
  data = DnD16
)

# Creating Log Odds Ratio
DnD17[, `:=`(
  num_comp_by_industry = sum(!is.na(capex)),
  invs_size_ind = sum(invest_dummy)
), by = .(treated, industry_simple, foreign_flag, tsid)]

DnD17 <- DnD17[, invs_size_ind_share := invs_size_ind / num_comp_by_industry] # Compute share of firms investing
DnD17 <- DnD17[, log_odds := fcase(
  invs_size_ind_share == 1 | log(invs_size_ind_share / (1 - invs_size_ind_share)) > 4, 4,
  invs_size_ind_share == 0 | log(invs_size_ind_share / (1 - invs_size_ind_share)) < -4, -4,
  invs_size_ind_share > 0 & invs_size_ind_share < 1, log(invs_size_ind_share / (1 - invs_size_ind_share))
)]  

model_DnD17_Ext <- feols(
  log_odds ~ treated * policy_period * foreign_flag + age + fte_log + log_tot_inc + log_lqdty | 
    # FIXED EFFECTS
    industry_simple^tsid_factor + 
    bg_id
  ,
  cluster = ~ industry_simple,
  data = DnD17
)

# Creating Log Odds Ratio
DnD18[, `:=`(
  num_comp_by_industry = sum(!is.na(capex)),
  invs_size_ind = sum(invest_dummy)
), by = .(treated, industry_simple, foreign_flag, tsid)]

DnD18 <- DnD18[, invs_size_ind_share := invs_size_ind / num_comp_by_industry] # Compute share of firms investing
DnD18 <- DnD18[, log_odds := fcase(
  invs_size_ind_share == 1 | log(invs_size_ind_share / (1 - invs_size_ind_share)) > 4, 4,
  invs_size_ind_share == 0 | log(invs_size_ind_share / (1 - invs_size_ind_share)) < -4, -4,
  invs_size_ind_share > 0 & invs_size_ind_share < 1, log(invs_size_ind_share / (1 - invs_size_ind_share))
)] 

model_DnD18_Ext <- feols(
  log_odds ~ treated * policy_period * foreign_flag + age + fte_log + log_tot_inc + log_lqdty | 
    # FIXED EFFECTS
    industry_simple^tsid_factor + 
    bg_id
  ,
  cluster = ~ industry_simple,
  data = DnD18
)

# Saving only coefficient outputs and not fixed effects
tidy_model_DnD15_Ext <- broom::tidy(model_DnD15_Ext)
tidy_model_DnD16_Ext <- broom::tidy(model_DnD16_Ext)
tidy_model_DnD17_Ext <- broom::tidy(model_DnD17_Ext)
tidy_model_DnD18_Ext <- broom::tidy(model_DnD18_Ext)

qs::qsave(tidy_model_DnD15_Ext, "P:/Company Tax/Econometrics for Clearance/Models to Clear/DDD With Controls/CAPEX_FRGN_DDD_15_Ext.qs")
qs::qsave(tidy_model_DnD16_Ext, "P:/Company Tax/Econometrics for Clearance/Models to Clear/DDD With Controls/CAPEX_FRGN_DDD_16_Ext.qs")
qs::qsave(tidy_model_DnD17_Ext, "P:/Company Tax/Econometrics for Clearance/Models to Clear/DDD With Controls/CAPEX_FRGN_DDD_17_Ext.qs")
qs::qsave(tidy_model_DnD18_Ext, "P:/Company Tax/Econometrics for Clearance/Models to Clear/DDD With Controls/CAPEX_FRGN_DDD_18_Ext.qs")

CAPEX_FRGN_DD_Extensive_num_obs_list <- data.frame(
  Model = model_names,
  Number_of_Observations = sapply(mget(model_names), nobs)
) %>% 
  mutate(Model = paste0("CAPEX_FRGN", Model))


# Intensive Regressions

#DnD

# Remove CAPEX = 0
DnD15 <- DnD15[capex != 0]
DnD16 <- DnD16[capex != 0]
DnD17 <- DnD17[capex != 0]
DnD18 <- DnD18[capex != 0]

## LIQUIDITY DDD
model_DnD15_Int <- feols(
  capex_log ~ treated * policy_period * lqdty_flag + age + fte_log + log_tot_inc + foreign_flag | 
    # FIXED EFFECTS
    industry_simple^tsid_factor + 
    bg_id,
  cluster = ~ bg_id, 
  data = DnD15)

model_DnD16_Int <- feols(
  capex_log ~ treated * policy_period * lqdty_flag + age + fte_log + log_tot_inc + foreign_flag | 
    # FIXED EFFECTS
    industry_simple^tsid_factor + bg_id,
  cluster = ~ bg_id, 
  data = DnD16)

model_DnD17_Int <- feols(
  capex_log ~ treated * policy_period * lqdty_flag  + age + fte_log + log_tot_inc + foreign_flag | 
    # FIXED EFFECTS
    industry_simple^tsid_factor + bg_id,
  cluster = ~ bg_id, 
  data = DnD17)

model_DnD18_Int <- feols(
  capex_log ~ treated * policy_period * lqdty_flag + age + fte_log + log_tot_inc + foreign_flag | 
    # FIXED EFFECTS
    industry_simple^tsid_factor + bg_id,
  cluster = ~ bg_id, 
  data = DnD18)

# Saving only coefficient outputs and not fixed effects
tidy_model_DnD15_Int <- broom::tidy(model_DnD15_Int)
tidy_model_DnD16_Int <- broom::tidy(model_DnD16_Int)
tidy_model_DnD17_Int <- broom::tidy(model_DnD17_Int)
tidy_model_DnD18_Int <- broom::tidy(model_DnD18_Int)

qs::qsave(tidy_model_DnD15_Int, "P:/Company Tax/Econometrics for Clearance/Models to Clear/DDD With Controls/CAPEX_LQDTY_DDD_15_Int.qs")
qs::qsave(tidy_model_DnD16_Int, "P:/Company Tax/Econometrics for Clearance/Models to Clear/DDD With Controls/CAPEX_LQDTY_DDD_16_Int.qs")
qs::qsave(tidy_model_DnD17_Int, "P:/Company Tax/Econometrics for Clearance/Models to Clear/DDD With Controls/CAPEX_LQDTY_DDD_17_Int.qs")
qs::qsave(tidy_model_DnD18_Int, "P:/Company Tax/Econometrics for Clearance/Models to Clear/DDD With Controls/CAPEX_LQDTY_DDD_18_Int.qs")

model_names <- c(
  "model_DnD15_Int", "model_DnD16_Int", "model_DnD17_Int", "model_DnD18_Int")

CAPEX_LQDTY_Intensive_num_obs_list <- data.frame(
  Model = model_names,
  Number_of_Observations = sapply(mget(model_names), nobs)
) %>% 
  mutate(Model = paste0("CAPEX_LQDTY_", Model))


## FOREIGN FLAG DDD

model_DnD15_Int <- feols(
  capex_log ~ treated * policy_period * foreign_flag + age + fte_log + log_tot_inc + log_lqdty | 
    # FIXED EFFECTS
    industry_simple^tsid_factor + 
    bg_id,
  cluster = ~ bg_id, 
  data = DnD15)

model_DnD16_Int <- feols(
  capex_log ~ treated * policy_period * foreign_flag + age + fte_log + log_tot_inc + log_lqdty | 
    # FIXED EFFECTS
    industry_simple^tsid_factor + bg_id,
  cluster = ~ bg_id, 
  data = DnD16)

model_DnD17_Int <- feols(
  capex_log ~ treated * policy_period * foreign_flag + age + fte_log + log_tot_inc + log_lqdty | 
    # FIXED EFFECTS
    industry_simple^tsid_factor + bg_id,
  cluster = ~ bg_id, 
  data = DnD17)

model_DnD18_Int <- feols(
  capex_log ~ treated * policy_period * foreign_flag + age + fte_log + log_tot_inc + log_lqdty | 
    # FIXED EFFECTS
    industry_simple^tsid_factor + bg_id,
  cluster = ~ bg_id, 
  data = DnD18)

# Saving only coefficient outputs and not fixed effects
tidy_model_DnD15_Int <- broom::tidy(model_DnD15_Int)
tidy_model_DnD16_Int <- broom::tidy(model_DnD16_Int)
tidy_model_DnD17_Int <- broom::tidy(model_DnD17_Int)
tidy_model_DnD18_Int <- broom::tidy(model_DnD18_Int)

qs::qsave(tidy_model_DnD15_Int, "P:/Company Tax/Econometrics for Clearance/Models to Clear/DDD With Controls/CAPEX_FRGN_DDD_15_Int.qs")
qs::qsave(tidy_model_DnD16_Int, "P:/Company Tax/Econometrics for Clearance/Models to Clear/DDD With Controls/CAPEX_FRGN_DDD_16_Int.qs")
qs::qsave(tidy_model_DnD17_Int, "P:/Company Tax/Econometrics for Clearance/Models to Clear/DDD With Controls/CAPEX_FRGN_DDD_17_Int.qs")
qs::qsave(tidy_model_DnD18_Int, "P:/Company Tax/Econometrics for Clearance/Models to Clear/DDD With Controls/CAPEX_FRGN_DDD_18_Int.qs")

model_names <- c(
  "model_DnD15_Int", "model_DnD16_Int", "model_DnD17_Int", "model_DnD18_Int")

CAPEX_FRGN_Intensive_num_obs_list <- data.frame(
  Model = model_names,
  Number_of_Observations = sapply(mget(model_names), nobs)
) %>% 
  mutate(Model = paste0("CAPEX_FRGN_", Model))


number_of_obs_df <- bind_rows(
  BAS_FRGN_Extensive_num_obs_list,
  BAS_FRGN_Intensive_num_obs_list,
  BAS_LQDTY_Extensive_num_obs_list,
  BAS_LQDTY_Intensive_num_obs_list,
  CAPEX_FRGN_DD_Extensive_num_obs_list,
  CAPEX_FRGN_Intensive_num_obs_list,
  CAPEX_LQDTY_DD_Extensive_num_obs_list,
  CAPEX_LQDTY_Intensive_num_obs_list
)


write_csv(number_of_obs_df, "P:/Company Tax/Econometrics for Clearance/Models to Clear/Number of Observations in each Model/DDD_w_ctrls_num_obs_list.csv")





